// Copyright 2001, 2002, 2003 Macromedia, Inc. All rights reserved.

//--------------------------------------------------------------------
// CLASS:
//   recordsetDialog
//
// DESCRIPTION:
//   Static class used to display the recordset SB UI. Determines which UI, 
//   simple or advanced, to display. Also, houses functionality shared between
//   the UI implementations and mediates switches between the UIs.
//   Note: this class documentation uses 'SBRecordset' to refer to your recordset
//   subclass of SBDatabaseCall.
//
// PUBLIC PROPERTIES:
//   none
//
// PUBLIC FUNCTIONS:
//   display(SBRecordset, cmdFilenameSimple, cmdFilenameAdvanced) - show the 
//     recordset dialog. Return updated recordset SB.
//--------------------------------------------------------------------

function recordsetDialog()
{
  // This is a static class and should not be instantiated.
  throw dwscripts.sprintf(MM.MSG_createStaticClass, "recordsetDialog");
}

// PUBLIC METHODS
recordsetDialog.display = recordsetDialog_display;

// PRIVATE METHODS
recordsetDialog.canDialogDisplayRecordset = recordsetDialog_canDialogDisplayRecordset;
recordsetDialog.getCommandDialogPref = recordsetDialog_getCommandDialogPref;
recordsetDialog.setCommandDialogPref = recordsetDialog_setCommandDialogPref;

// Following functions are used by the recordset dialogs
recordsetDialog.onClickOK = recordsetDialog_onClickOK;
recordsetDialog.onClickCancel = recordsetDialog_onClickCancel;
recordsetDialog.onClickSwitchUI = recordsetDialog_onClickSwitchUI;
recordsetDialog.displayTestDialog = recordsetDialog_displayTestDialog;

// CLASS CONSTANTS
// Availabe recordset UI actions. 
recordsetDialog.UI_ACTION_OK            = 1 << 0;
recordsetDialog.UI_ACTION_CANCEL        = 1 << 1;
recordsetDialog.UI_ACTION_SWITCH_SIMPLE = 1 << 2;
recordsetDialog.UI_ACTION_SWITCH_ADV    = 1 << 3;

// Preference information.
recordsetDialog.CMD_FILENAME_PREF_SECTION = "Extensions\\ServerBehaviors\\Recordset";
recordsetDialog.CMD_FILENAME_PREF_KEY = "Command Filename";

// Name used for the parameter in a simple recordset.
recordsetDialog.SIMPLE_PARAM_NAME = "param1";

//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.display
//
// DESCRIPTION:
//   Show the recordset dialog. Determines which dialog to present based on
//   the recordset and the previous dialog used. Retrieves updated values for the
//   recordset.
//   This function invokes the recordset commands with an array argument object. The
//   first argument is SBRecordset and the second is the command filename for the
//   other dialog in the pair (if invoking the simple dialog, this argument would be
//   the advanced dialog filename). 
//
// ARGUMENTS:
//   priorSBRecordset - SBRecordset. Existing recordset SB. If there is no existing
//     recordset, send in an empty instance of a SBDatabaseCall subclass.
//   cmdFilenameSimple - string. Filename of the command to use for the
//     simple version of the recordset dialog.
//   cmdFilenameAdvanced - string. Filename of the command to use for the
//     advanced version of the recordset dialog. 
//
// RETURNS:
//   SBRecordset - updated recordset. null if cancelled.
//--------------------------------------------------------------------

function recordsetDialog_display(priorSBRecordset,
                                 cmdFilenameSimple, 
                                 cmdFilenameAdvanced)
{
  // Last action in command dialog prior to return
  var uiAction = recordsetDialog.UI_ACTION_SWITCH;

  // Command filename of recordset dialog currently displayed.
  var cmdFilename = recordsetDialog.getCommandDialogPref(cmdFilenameSimple);
  if (   cmdFilename != cmdFilenameSimple
      && cmdFilename != cmdFilenameAdvanced
     )
  {
    cmdFilename = cmdFilenameSimple;
  }

  var newSBRecordset = priorSBRecordset.makeEditableCopy();

  // If the simple dialog is the preference, but the simple dialog cannot display
  //   the recordset, override the preference.
  if (cmdFilename == cmdFilenameSimple)
  {
    if (!recordsetDialog.canDialogDisplayRecordset(cmdFilenameSimple, priorSBRecordset))
    { 
      // Note: We are assuming that we will always be able to use the advanced 
      //   dialog.
      cmdFilename = cmdFilenameAdvanced;
    }
  }
  
  // The simple/advanced recordset UI loop
  while (   uiAction != recordsetDialog.UI_ACTION_CANCEL 
         && uiAction != recordsetDialog.UI_ACTION_OK
        ) 
  {
    var cmdFilenameForSwitch = (cmdFilename == cmdFilenameSimple) ? 
                               cmdFilenameAdvanced : cmdFilenameSimple;

    // Note newSBRecordset will be updated using pass by reference.
    uiAction = dwscripts.callCommand(cmdFilename, [newSBRecordset, cmdFilenameForSwitch]);

    // Update the command file to use if we are switching.
    switch (uiAction)
    {
      case recordsetDialog.UI_ACTION_SWITCH_SIMPLE:
        cmdFilename = cmdFilenameSimple;
        break;
      case recordsetDialog.UI_ACTION_SWITCH_ADV:
        cmdFilename = cmdFilenameAdvanced;
        break;
      case recordsetDialog.UI_ACTION_OK:
        // noop
        break;
      default:
        // User cancelled. In case user clicked the 'X' button, set uiAction to
        //   cancelled since it won't be set.
        uiAction = recordsetDialog.UI_ACTION_CANCEL;
        break;        
    }
  }

  return (uiAction == recordsetDialog.UI_ACTION_OK) ? newSBRecordset : null;
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.onClickOK
//
// DESCRIPTION:
//   Event handler for the click of the OK button in a recordset dialog. Recordset
//   dialogs call this function from their onClickOK handlers. If the recordset
//   is valid, closes the window and sets the return value for the command.
//   Otherwise, displays an error message.
//
// ARGUMENTS:
//   theWindow - window. window object for the calling command dialog.
//   sbRecordset - SBRecordset. Recordset info updated from the dialog.
//
// RETURNS:
//   none
//--------------------------------------------------------------------

function recordsetDialog_onClickOK(theWindow, sbRecordset, cmdFileName)
{
  if (!sbRecordset.checkData(false))
  {
    alert(sbRecordset.getErrorMessage());
  }
  else
  {
    dwscripts.setCommandReturnValue(recordsetDialog.UI_ACTION_OK);
    // Store the dialog used last.
    recordsetDialog.setCommandDialogPref(cmdFileName);

    theWindow.close();
  }
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.onClickCancel
//
// DESCRIPTION:
//   Event handler for the click of the Cancel button in a recordset dialog. Recordset
//   dialogs call this function from their onClickCancel handlers. If everything
//   is in order, closes the window and sets the return value for the command.
//
// ARGUMENTS:
//   theWindow - window. window object for the calling command dialog.
//
// RETURNS:
//   none
//--------------------------------------------------------------------

function recordsetDialog_onClickCancel(theWindow)
{
  // No need to do any additional processing for now.
  dwscripts.setCommandReturnValue(recordsetDialog.UI_ACTION_CANCEL);
  theWindow.close();
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.onClickSwitchUI
//
// DESCRIPTION:
//   Mediate a switch between recordset dialogs. This function is called by a 
//   recordset dialog when the user wants to switch the UI. It handles closing
//   the window and setting the return value for the command if it is a valid switch.
//   Otherwise, displays an error message.
//
// ARGUMENTS:
//   theWindow - window. Window object for the calling command dialog.
//   uiAction - enumerated. One of recordsetDialog.UI_ACTION_SWITCH*.
//   sbRecordset - SBRecordset. Recordset updated from the dialog. 
//   switchCmdFilename - string. Command filename for the recordset dialog to switch
//     to.
//
// RETURNS:
//   nothing
//--------------------------------------------------------------------

function recordsetDialog_onClickSwitchUI(theWindow, uiAction, sbRecordset, switchCmdFilename)
{
  switch (uiAction)
  {
    case recordsetDialog.UI_ACTION_SWITCH_SIMPLE:
      // If the recordset is not simple, send control back to the calling dialog.
      if (!recordsetDialog.canDialogDisplayRecordset(switchCmdFilename, sbRecordset)) 
      {
        alert(dwscripts.sprintf(MM.MSG_SQLNotSimple, dwscripts.getRecordsetDisplayName()));
        return;
      }
      break;
    case recordsetDialog.UI_ACTION_SWITCH_ADV:
      // Assume switching to advanced is always ok. 
      break;
    default:
      // Invalid dialog. Send control back to the calling dialog
      return;
  }
  
  // Set return value and close the window for the command dialog.
  dwscripts.setCommandReturnValue(uiAction);
  theWindow.close();
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.displayTestDialog
//
// DESCRIPTION:
//   If the recordset info is valid, display the query results in a dialog.
//
// ARGUMENTS:
//   sbRecordset - SBRecordset. Recordset info updated from the dialog.
//
// RETURNS:
//   none
//--------------------------------------------------------------------

function recordsetDialog_displayTestDialog(sbRecordset)
{
  if (!sbRecordset.checkData(true))
  {
    alert(sbRecordset.getErrorMessage());
    return;
  }

  var theSQL = sbRecordset.getSQLForTest();
  
  if (theSQL)
  {
    MMDB.showResultset(sbRecordset.getConnectionName(), theSQL);
  }
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.getCommandDialogPref
//
// DESCRIPTION:
//   Get the preference for the command file to use for the recordset dialog.
//
// ARGUMENTS:
//   cmdFilenameDefault - string. Default command dialog filename to display
//     the recordset.
//
// RETURNS:
//   String - command file name. One of recordsetDialog.CMD_FILENAME_*
//            returns simple command dialog by default
//--------------------------------------------------------------------

function recordsetDialog_getCommandDialogPref(cmdFilenameDefault)
{
  prefValue = dw.getPreferenceString(recordsetDialog.CMD_FILENAME_PREF_SECTION, 
                                     recordsetDialog.CMD_FILENAME_PREF_KEY, 
                                     cmdFilenameDefault);

  return prefValue;
}


//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.setCommandDialogPref
//
// DESCRIPTION:
//   Set the preference for the command file to use for the recordset dialog.
//
// ARGUMENTS:
//   cmdFilename - string. Command dialog filename. One of
//     recordsetDialog.CMD_FILENAME_*.
//
// RETURNS:
//   none
//--------------------------------------------------------------------

function recordsetDialog_setCommandDialogPref(cmdFilename)
{
  dw.setPreferenceString(recordsetDialog.CMD_FILENAME_PREF_SECTION, 
                         recordsetDialog.CMD_FILENAME_PREF_KEY, 
                         cmdFilename
                        );  
}

//--------------------------------------------------------------------
// FUNCTION:
//   recordsetDialog.canDialogDisplayRecordset
//
// DESCRIPTION:
//   Determines if the recordset command dialog can accept the recordset server
//   behavior. Check if the function 'canDisplayRecordset()' is defined in the command
//   dialog. If it is defined, invoke it to determine if the command can 
//   accept the recordset. If it is not defined, assume the recordset can be
//   displayed.
//
// ARGUMENTS:
//   cmdFilename - string. name of command dialog.
//   sbRecordset - SBRecordset. Recordset instance to check.
//
// RETURNS:
//   boolean. true if command can accept the SBRecordset.
//--------------------------------------------------------------------

function recordsetDialog_canDialogDisplayRecordset(cmdFilename, sbRecordset)
{  
  var bCanDisplayRecordset = false; 
  var domCommand = null;
  var windowCommand = null;
  
  if (cmdFilename)
  {
    domCommand = dw.getDocumentDOM(dw.getConfigurationPath() + "/Commands/" + cmdFilename); 
    if (domCommand)
    {
      bCanDisplayRecordset = true;
      windowCommand = domCommand.parentWindow;
      if (   windowCommand.canDisplayRecordset
          && !windowCommand.canDisplayRecordset(sbRecordset)
         )
      {
        bCanDisplayRecordset = false;
      }
    }
  }
  
  return bCanDisplayRecordset;
}


